using System;
using System.Collections.Generic;
using ORTS.Common;
using ORTS.Scripting.Api;
using System.Text;
using System.Linq;
using Event = Orts.Common.Event;
namespace ORTS.Scripting.Script
{
    public class T250PowerSupply : ElectricPowerSupply
    {
        bool DualMode = false;
        bool IsRear = false;
        bool IsRearUnit = false;
        bool IsMultipleUnit = false;
        int RemoteHeadIndex;
        int LocalCETIndex;
        int RemoteCETIndex;
        enum PowerMode
        {
            AC25kV,
            DC3kV,
            Diesel,
        }
        enum NeutralSectionState
        {
            Inactive,
            Preparation,
            OpenCB,
            Active,
            WaitVoltage,
        }
        NeutralSectionState CurrentNeutralSectionState;
        bool NeutralSectionByTCS;

        enum PantographDownSectionState
        {
            Inactive,
            Preparation,
            OpenCB,
            LowerPantograph,
            Active,
            RaisePantograph,
        }
        PantographDownSectionState currentPantographDownSectionState;
        PantographDownSectionState CurrentPantographDownSectionState
        {
            get
            {
                return currentPantographDownSectionState;
            }
            set
            {
                if (currentPantographDownSectionState == value) return;
                currentPantographDownSectionState = value;
                switch (currentPantographDownSectionState)
                {
                    case PantographDownSectionState.LowerPantograph:
                    case PantographDownSectionState.RaisePantograph:
                        HandlePantographs();
                        break;
                }
            }
        }
        enum TractionSystemChangeState
        {
            Inactive,
            Preparation,
            Prepared,
            WaitNeutral,
            OpenCB,
            LowerOldPantograph,
            RaiseNewPantograph,
            Active,
            WaitVoltage,
            WaitEnd,
            AuxiliaryFunction,
            ReconfigureLower,
            ReconfigureRaise,
        }
        enum PantographSelector
        {
            Rear,
            Front,
            Both,
            Automatic,
        }

        public enum CustomEvent
        {
            LowerPantographTCS,
            RaisePantographTCS,
            StartNeutralSection,
            EndNeutralSection,
            StartNeutralSectionTCS,
            EndNeutralSectionTCS,
        }

        PantographSelector PantographSelection = PantographSelector.Automatic;
        TractionSystemChangeState currentTractionSystemChangeState;
        TractionSystemChangeState CurrentTractionSystemChangeState
        {
            get
            {
                return currentTractionSystemChangeState;
            }
            set
            {
                if (currentTractionSystemChangeState == value) return;
                currentTractionSystemChangeState = value;
                switch (currentTractionSystemChangeState)
                {
                    case TractionSystemChangeState.Preparation:
                    case TractionSystemChangeState.ReconfigureLower:
                    case TractionSystemChangeState.ReconfigureRaise:
                        HandlePantographs();
                        break;
                    case TractionSystemChangeState.RaiseNewPantograph:
                        if (!IsRear) HandlePantographs();
                        break;
                    case TractionSystemChangeState.LowerOldPantograph:
                        if (IsRear) HandlePantographs();
                        break;
                }
            }
        }
        bool TractionSystemChangeDefrost;
        OdoMeter LocomotivePositionDelay;
        bool Flipped;
        int LeadIndex;
        private bool pantoUp;
        bool PantoUp
        {
            get
            {
                return pantoUp;
            }
            set
            {
                if (pantoUp == value) return;
                pantoUp = value;
                HandlePantographs();
            }
        }

        int PantoTarget;
        bool LocalPantoUp;

        class PowerContactor
        {
            Timer ChangeTimer;
            private CircuitBreakerState state;
            public CircuitBreakerState TargetState;
            public CircuitBreakerState State
            {
                get
                {
                    if (ChangeTimer.Triggered)
                    {
                        state = TargetState;
                        ChangeTimer.Stop();
                    }
                    return state;
                }
                set
                {
                    if (TargetState != value)
                    {
                        TargetState = value;
                        ChangeTimer.Start();
                        //if (TargetState == CircuitBreakerState.Closed) state = CircuitBreakerState.Closing;
                    }
                }
            }
            public PowerContactor(ElectricPowerSupply supply)
            {
                ChangeTimer = new Timer(supply);
                ChangeTimer.Setup(5);
            }
        }
        PowerContactor LocalRoofLineContactor;
        PowerContactor LocalTrainLineContactor;

        CircuitBreakerState LocalRoofLineContactorState
        {
            get
            {
                return LocalRoofLineContactor.State;
            }
            set
            {
                LocalRoofLineContactor.State = value;
            }
        }
        CircuitBreakerState RemoteRoofLineContactorState;

        CircuitBreakerState LocalTrainLineContactorState
        {
            get
            {
                return LocalTrainLineContactor.State;
            }
            set
            {
                LocalTrainLineContactor.State = value;
            }
        }
        CircuitBreakerState RemoteTrainLineContactorState;
        private CircuitBreakerState rearCircuitBreakerState;
        private CircuitBreakerState localCETCircuitBreakerState;
        private CircuitBreakerState rearCETCircuitBreakerState;
        CircuitBreakerState LocalCircuitBreakerState;
        CircuitBreakerState RemoteCircuitBreakerState
        {
            get
            {

                if (
                    (CurrentPowerMode == PowerMode.AC25kV
                        && (LocalRoofLineContactorState != CircuitBreakerState.Closed || RemoteRoofLineContactorState != CircuitBreakerState.Closed))
                    || (CurrentPowerMode == PowerMode.DC3kV
                        && (LocalTrainLineContactorState != CircuitBreakerState.Closed || RemoteTrainLineContactorState != CircuitBreakerState.Closed))
                    || CurrentPowerMode == PowerMode.Diesel
                )
                {
                    return CircuitBreakerState.Open;
                }
                return rearCircuitBreakerState;
            }
            set
            {
                rearCircuitBreakerState = value;
            }
        }
        CircuitBreakerState LocalCETCircuitBreakerState
        {
            get
            {
                if (CurrentPowerMode != PowerMode.Diesel || LocalTrainLineContactorState != CircuitBreakerState.Closed)
                {
                    return CircuitBreakerState.Open;
                }
                return localCETCircuitBreakerState;
            }
            set
            {
                localCETCircuitBreakerState = value;
            }
        }
        CircuitBreakerState RemoteCETCircuitBreakerState
        {
            get
            {
                if (CurrentPowerMode != PowerMode.Diesel || LocalTrainLineContactorState != CircuitBreakerState.Closed)
                {
                    return CircuitBreakerState.Open;
                }
                return rearCETCircuitBreakerState;
            }
            set
            {
                rearCETCircuitBreakerState = value;
            }
        }
        PowerSupplyState LocalCETGeneratorState;
        PowerSupplyState RemoteCETGeneratorState;
        float LocalCETPowerW;
        float RemoteCETPowerW;
        float LocalPantographVoltageV;
        float RemotePantographVoltageV;

        PowerMode TargetPowerMode;
        private PowerMode CurrentPowerMode = PowerMode.DC3kV;

        Timer PowerOnTimer;
        Timer AuxPowerOnTimer;
        Timer DieselConfigureTimer;

        IIRFilter PantographFilter;
        IIRFilter VoltageFilter;

        int prevNumberOfLocomotives;
        SortedDictionary<int,int> ConnectedLocomotives = new SortedDictionary<int,int>();

        bool QuickPowerOn;
        bool LocalCabActive;
        bool RemoteCabActive;
        bool TargetParkingMode;
        bool ParkingMode
        {
            get
            {
                return ServiceRetentionActive;
            }
            set
            {
                ServiceRetentionActive = value;
            }
        }

        enum IndicatorStatus
        {
            Off,
            On,
            Flash,
            CounterFlash
        }
        Dictionary<int, IndicatorStatus> Indicators = new Dictionary<int, IndicatorStatus>();
        Blinker Blink;

        public override void Initialize()
        {
            PantographFilter = new IIRFilter(IIRFilter.FilterTypes.Butterworth, 1, IIRFilter.HzToRad(0.7f), 0.001f);
            VoltageFilter = new IIRFilter(IIRFilter.FilterTypes.Butterworth, 1, IIRFilter.HzToRad(0.7f), 0.001f);

            LocalTrainLineContactor = new PowerContactor(this);
            LocalRoofLineContactor = new PowerContactor(this);

            PowerOnTimer = new Timer(this);
            PowerOnTimer.Setup(PowerOnDelayS());
            
            AuxPowerOnTimer = new Timer(this);
            AuxPowerOnTimer.Setup(AuxPowerOnDelayS());

            DieselConfigureTimer = new Timer(this);
            DieselConfigureTimer.Setup(5);

            Blink = new Blinker(this);
            Blink.Setup(1);
            Blink.Start();
            
            RestoreVoltageOdometer = new OdoMeter(this);
            RestoreVoltageOdometer.Setup(50);
            
            LocomotivePositionDelay = new OdoMeter(this);

            if (SimulatedLineVoltageV() > 10000) CurrentPowerMode = PowerMode.AC25kV;
            else if (SimulatedLineVoltageV() > 1900) CurrentPowerMode = PowerMode.DC3kV;
            else CurrentPowerMode = PowerMode.Diesel;
            TargetPowerMode = CurrentPowerMode;

            SetCustomizedCabviewControlName(0, "Preparación cambio de sistema");
            SetCustomizedCabviewControlName(1, "Cambio de sistema");
            SetCustomizedCabviewControlName(2, "Cambio catenaria/diésel");
            SetCustomizedCabviewControlName(3, "Zona neutra");
            SetCustomizedCabviewControlName(4, "Activación sistema CA");
            SetCustomizedCabviewControlName(5, "Activación sistema CC");
            SetCustomizedCabviewControlName(6, "Arranque motores diésel");
            SetCustomizedCabviewControlName(7, "Paro motores diésel");
            SetCustomizedCabviewControlName(8, "Cambio sistema función auxiliar");

            for (int i=0; i<=7; i++)
            {
                Indicators[i] = IndicatorStatus.Off;
            }

            SignalEventToTcsWithMessage(PowerSupplyEvent.OpenCircuitBreaker, "AUTOMATIC_NEUTRAL_SECTION");
        }
        public override void Update(float elapsedClockSeconds)
        {
            if (prevNumberOfLocomotives != NumberOfLocomotives())
            {
                prevNumberOfLocomotives = NumberOfLocomotives();
                ConnectedLocomotives.Clear();
                ConnectedLocomotives[IndexOfLocomotive()] = 130;
                if (IndexOfLocomotive() + 1 < NumberOfLocomotives()) SignalEventToOtherLocomotive(IndexOfLocomotive() + 1, PowerSupplyEvent.SwitchOnElectricTrainSupply, 130);
                if (IndexOfLocomotive() > 0) SignalEventToOtherLocomotive(IndexOfLocomotive() - 1, PowerSupplyEvent.SwitchOnElectricTrainSupply, 130);
                LeadIndex = -1;
                RemoteCabActive = false;
            }
            {
                int headno=0;
                int cetno=0;
                int unitno=0;
                int ownunit=-1;
                int ownno=-1;
                int leadunit=-1;
                int leadno=-1;
                foreach (var kvp in ConnectedLocomotives)
                {
                    if (headno > 1)
                    {
                        unitno++;
                        headno = cetno = 0;
                    }
                    if (kvp.Key == LeadIndex)
                    {
                        leadunit = unitno;
                        leadno = headno;
                    }
                    if (kvp.Key == IndexOfLocomotive())
                    {
                        ownunit = unitno;
                        ownno = headno;
                        Flipped = headno == 1;
                    }
                    if (kvp.Value == 130) headno++;
                    else if (kvp.Value == 730) cetno++;
                }
                IsRear = leadno != ownno;
                IsRearUnit = ownunit != leadunit;
                IsMultipleUnit = unitno > 0;
                headno=0;
                cetno=0;
                unitno=0;
                LocalCETIndex = -1;
                RemoteCETIndex = -1;
                RemoteHeadIndex = -1;
                foreach (var kvp in ConnectedLocomotives)
                {
                    if (headno > 1)
                    {
                        unitno++;
                        headno = cetno = 0;
                    }
                    if (unitno == ownunit)
                    {
                        if (kvp.Value == 130 && headno != ownno) RemoteHeadIndex = kvp.Key;
                        if (kvp.Value == 730)
                        {
                            if (ownno == cetno) LocalCETIndex = kvp.Key;
                            else RemoteCETIndex = kvp.Key;
                        }
                    }
                    if (kvp.Value == 130) headno++;
                    else if (kvp.Value == 730) cetno++;
                }
                DualMode = LocalCETIndex >= 0;
            }
            
            switch (CurrentTractionSystemChangeState)
            {
                case TractionSystemChangeState.Preparation:
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        if ((IsRear && LocalCETCircuitBreakerState == CircuitBreakerState.Closed) || (!IsRear && LocalTrainLineContactorState == CircuitBreakerState.Open))
                            CurrentTractionSystemChangeState = TractionSystemChangeState.Prepared;
                    }
                    else if (IsRear)
                    {
                        if (LocalPantoUp)
                        {
                            if (LocalCircuitBreakerState == CircuitBreakerState.Open && CircuitBreakerClosingAuthorization())
                                SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOn);
                            if (LocalCircuitBreakerState == CircuitBreakerState.Closed)
                                CurrentTractionSystemChangeState = TractionSystemChangeState.Prepared;
                        }
                    }
                    else
                    {
                        if (CurrentPantographState() == PantographState.Down && LocalRoofLineContactorState == CircuitBreakerState.Open && LocalTrainLineContactorState == CircuitBreakerState.Open)
                            CurrentTractionSystemChangeState = TractionSystemChangeState.Prepared;
                    }
                    break;
                case TractionSystemChangeState.Prepared:
                    break;
                case TractionSystemChangeState.WaitNeutral:
                    if (LocomotivePositionDelay.Triggered)
                    {
                        if (IsRear) CurrentTractionSystemChangeState = TractionSystemChangeState.OpenCB;
                        else if (CurrentPowerMode == PowerMode.Diesel) CurrentTractionSystemChangeState = TractionSystemChangeState.Active;
                        else CurrentTractionSystemChangeState = TractionSystemChangeState.RaiseNewPantograph;
                    }
                    break;
                case TractionSystemChangeState.OpenCB:
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        if (LocalTrainLineContactorState == CircuitBreakerState.Open)
                            CurrentTractionSystemChangeState = TractionSystemChangeState.WaitEnd;
                    }
                    else if (LocalCircuitBreakerState != CircuitBreakerState.Open)
                        SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
                    else if (LocalCircuitBreakerState == CircuitBreakerState.Open)
                        CurrentTractionSystemChangeState = TractionSystemChangeState.LowerOldPantograph;
                    break;
                case TractionSystemChangeState.LowerOldPantograph:
                    if (CurrentPantographState() == PantographState.Down)
                        CurrentTractionSystemChangeState = TractionSystemChangeState.WaitEnd;
                    break;
                case TractionSystemChangeState.RaiseNewPantograph:
                    if (LocalPantoUp)
                        CurrentTractionSystemChangeState = TractionSystemChangeState.Active;
                    break;
                case TractionSystemChangeState.Active:
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        CurrentTractionSystemChangeState = TractionSystemChangeState.WaitEnd;
                    }
                    else if (LocalPantographVoltageV < 1900)
                    {
                        CurrentTractionSystemChangeState = TractionSystemChangeState.WaitVoltage;
                    }
                    break;
                case TractionSystemChangeState.WaitVoltage:
                    if ((TargetPowerMode == PowerMode.DC3kV && LocalPantographVoltageV > 1900 && LocalPantographVoltageV < 4250) ||
                        (TargetPowerMode == PowerMode.AC25kV && LocalPantographVoltageV > 16000 && LocalPantographVoltageV < 31000)
                    )
                    {
                        CurrentPowerMode = TargetPowerMode;
                        CurrentTractionSystemChangeState = TractionSystemChangeState.WaitEnd;
                        SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOn);
                    }
                    break;
                case TractionSystemChangeState.WaitEnd:
                    if (CurrentPowerMode != PowerMode.Diesel && CurrentPowerMode != TargetPowerMode) CurrentPowerMode = TargetPowerMode;
                    break;
                case TractionSystemChangeState.ReconfigureLower:
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        CurrentTractionSystemChangeState = TractionSystemChangeState.ReconfigureRaise;
                    }
                    else
                    {
                        CurrentPowerMode = TargetPowerMode;
                        if (IsRear)
                        {
                            CurrentTractionSystemChangeState = TractionSystemChangeState.ReconfigureRaise;
                        }
                        else
                        {
                            if (CurrentPantographState() == PantographState.Down)
                                CurrentTractionSystemChangeState = TractionSystemChangeState.ReconfigureRaise;
                        }
                    }
                    break;
                case TractionSystemChangeState.ReconfigureRaise:
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        if (LocalTrainLineContactorState == CircuitBreakerState.Closed)
                        {
                            CurrentTractionSystemChangeState = TractionSystemChangeState.Inactive;
                            TractionSystemChangeDefrost = false;
                        }
                    }
                    else if (LocalPantoUp)
                    {
                        if (CircuitBreakerClosingAuthorization())
                        {
                            SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOn);
                            CurrentTractionSystemChangeState = TractionSystemChangeState.Inactive;
                            TractionSystemChangeDefrost = false;
                        }
                    }
                    else if (CurrentPantographState() == PantographState.Up)
                    {
                        if ((CurrentPowerMode == PowerMode.AC25kV && LocalRoofLineContactorState == CircuitBreakerState.Closed && RemoteRoofLineContactorState == CircuitBreakerState.Closed)
                            || (CurrentPowerMode == PowerMode.DC3kV && LocalTrainLineContactorState == CircuitBreakerState.Closed && RemoteTrainLineContactorState == CircuitBreakerState.Closed)
                        )
                        {
                            CurrentTractionSystemChangeState = TractionSystemChangeState.Inactive;
                            TractionSystemChangeDefrost = false;
                        }
                    }
                    break;
            }
            if (CurrentTractionSystemChangeState == TractionSystemChangeState.AuxiliaryFunction)
            {
                Indicators[0] = Indicators[1] = IndicatorStatus.Flash;
                Indicators[8] = IndicatorStatus.On;
            }
            else
            {
                if (CurrentTractionSystemChangeState == TractionSystemChangeState.Preparation || CurrentTractionSystemChangeState >= TractionSystemChangeState.ReconfigureLower) Indicators[0] = IndicatorStatus.Flash;
                else if (CurrentTractionSystemChangeState == TractionSystemChangeState.Prepared) Indicators[0] = IndicatorStatus.On;
                else Indicators[0] = IndicatorStatus.Off;
                if (CurrentTractionSystemChangeState > TractionSystemChangeState.Prepared && CurrentTractionSystemChangeState <= TractionSystemChangeState.WaitVoltage) Indicators[1] = IndicatorStatus.Flash;
                else if (CurrentTractionSystemChangeState == TractionSystemChangeState.WaitEnd) Indicators[1] = IndicatorStatus.On;
                else Indicators[1] = IndicatorStatus.Off;
                Indicators[8] = IndicatorStatus.Off;
            }
            switch (CurrentNeutralSectionState)
            {
                case NeutralSectionState.Preparation:
                    if (LocomotivePositionDelay.Triggered)
                    {
                        CurrentNeutralSectionState = NeutralSectionState.OpenCB;
                        LocomotivePositionDelay.Stop();
                    }
                    break;
                case NeutralSectionState.OpenCB:
                    if (LocalCircuitBreakerState != CircuitBreakerState.Open)
                        SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
                    else if (LocalCircuitBreakerState == CircuitBreakerState.Open)
                        CurrentNeutralSectionState = NeutralSectionState.Active;
                    break;
                case NeutralSectionState.Active:
                    if (PantographVoltageV() < 1900)
                        CurrentNeutralSectionState = NeutralSectionState.WaitVoltage;
                    if (LocalCircuitBreakerState != CircuitBreakerState.Open)
                        SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
                    break;
                case NeutralSectionState.WaitVoltage:
                    if (CircuitBreakerClosingAuthorization() && (!NeutralSectionByTCS || LocomotivePositionDelay.Triggered))
                    {
                        if (LocalCircuitBreakerState == CircuitBreakerState.Open)
                        {
                            SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOn);
                            CurrentNeutralSectionState = NeutralSectionState.Inactive;
                        }
                    }
                    else if (LocalCircuitBreakerState != CircuitBreakerState.Open)
                    {
                        SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
                    }
                    if (CurrentCircuitBreakerState() == CircuitBreakerState.Closed)
                    {
                        CurrentNeutralSectionState = NeutralSectionState.Inactive;
                    }
                    break;
            }
            if (CurrentCircuitBreakerState() != CircuitBreakerState.Closed) MaxThrottlePercent = 0;
            else if (CurrentNeutralSectionState != NeutralSectionState.Inactive)
            {
                if (CurrentNeutralSectionState == NeutralSectionState.WaitVoltage)
                {
                    MaxThrottlePercent = 100;
                }
                else
                {
                    float offset;
                    if (LocalCircuitBreakerState == CircuitBreakerState.Closed) offset = 0;
                    else if (IsRear) offset = -158;
                    else offset = 158;
                    MaxThrottlePercent = Math.Max(Math.Min(((LocomotivePositionDelay.RemainingValue+offset)/Math.Max(SpeedMpS(), 10)/3.0f-1)*100, 100), 0);
                }
            }
            else MaxThrottlePercent = 100;
            if (CurrentNeutralSectionState != NeutralSectionState.Inactive) Indicators[3] = IndicatorStatus.On;
            else Indicators[3] = IndicatorStatus.Off;
            switch (CurrentPantographDownSectionState)
            {
                case PantographDownSectionState.Preparation:
                    if (LocomotivePositionDelay.Triggered)
                        CurrentPantographDownSectionState = PantographDownSectionState.OpenCB;
                    break;
                case PantographDownSectionState.OpenCB:
                    if (LocalCircuitBreakerState != CircuitBreakerState.Open)
                        SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
                    else if (LocalCircuitBreakerState == CircuitBreakerState.Open)
                        CurrentPantographDownSectionState = PantographDownSectionState.LowerPantograph;
                    break;
                case PantographDownSectionState.LowerPantograph:
                    if (CurrentPantographState() == PantographState.Down)
                    {
                        CurrentPantographDownSectionState = PantographDownSectionState.Active;
                        LocomotivePositionDelay.Stop();
                    }
                    break;
                case PantographDownSectionState.Active:
                    if (TargetPowerMode != PowerMode.Diesel) CurrentPowerMode = TargetPowerMode;
                    if (LocomotivePositionDelay.Triggered) CurrentPantographDownSectionState = PantographDownSectionState.RaisePantograph;
                    break;
                case PantographDownSectionState.RaisePantograph:
                    if (LocalPantoUp)
                    {
                        if (CircuitBreakerClosingAuthorization())
                        {
                            SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOn);
                            CurrentPantographDownSectionState = PantographDownSectionState.Inactive;
                        }
                    }
                    else if (CurrentPantographState() == PantographState.Up)
                    {
                        CurrentPantographDownSectionState = PantographDownSectionState.Inactive;
                    }
                    break;
            }
            if (!PantoUp) CurrentPantographDownSectionState = PantographDownSectionState.Inactive;
            if (!PantoUp && CurrentPowerMode != TargetPowerMode && CurrentPowerMode != PowerMode.Diesel && TargetPowerMode != PowerMode.Diesel)
            {
                CurrentPowerMode = TargetPowerMode;
            }
            if (TargetPowerMode == PowerMode.AC25kV) Indicators[4] = CurrentPowerMode == PowerMode.AC25kV ? IndicatorStatus.On : IndicatorStatus.Flash;
            else Indicators[4] = IndicatorStatus.Off;
            if (TargetPowerMode == PowerMode.DC3kV) Indicators[5] = CurrentPowerMode == PowerMode.DC3kV ? IndicatorStatus.On : IndicatorStatus.Flash;
            else Indicators[5] = IndicatorStatus.Off;
            if (CurrentPowerMode == PowerMode.Diesel)
            {
                CurrentNeutralSectionState = NeutralSectionState.Inactive;
                CurrentPantographDownSectionState = PantographDownSectionState.Inactive;
            }
            if (CurrentPowerMode == PowerMode.Diesel || TargetPowerMode == PowerMode.Diesel) PantoUp = false;
            if (CurrentPowerMode == PowerMode.Diesel && TargetPowerMode != PowerMode.Diesel)
            {
                if (LocalCETCircuitBreakerState == CircuitBreakerState.Open)
                {
                    if (!DieselConfigureTimer.Started) DieselConfigureTimer.Start();
                    if (DieselConfigureTimer.Triggered) CurrentPowerMode = TargetPowerMode;
                }
            }
            else if (CurrentPowerMode != PowerMode.Diesel && TargetPowerMode == PowerMode.Diesel)
            {
                if (LocalCircuitBreakerState == CircuitBreakerState.Open && CurrentPantographState() == PantographState.Down)
                {
                    if (!DieselConfigureTimer.Started) DieselConfigureTimer.Start();
                    if (DieselConfigureTimer.Triggered
                        && LocalTrainLineContactorState == CircuitBreakerState.Closed && LocalRoofLineContactorState == CircuitBreakerState.Open
                        && (LocalCETGeneratorState == PowerSupplyState.PowerOn || RemoteCETGeneratorState == PowerSupplyState.PowerOn))
                    {
                        CurrentPowerMode = PowerMode.Diesel;
                    }
                }
            }
            else if (DieselConfigureTimer.Started) DieselConfigureTimer.Stop();
            if (CurrentPowerMode != TargetPowerMode && (CurrentPowerMode == PowerMode.Diesel || TargetPowerMode == PowerMode.Diesel)) Indicators[2] = IndicatorStatus.Flash;
            else if (CurrentPowerMode == PowerMode.Diesel) Indicators[2] = IndicatorStatus.On;
            else Indicators[2] = IndicatorStatus.Off;
            switch (CurrentHelperEnginesState())
            {
                case DieselEngineState.Unavailable:
                case DieselEngineState.Stopped:
                    Indicators[6] = Indicators[7] = IndicatorStatus.Off;
                    break;
                case DieselEngineState.Starting:
                    Indicators[6] = IndicatorStatus.Flash;
                    Indicators[7] = IndicatorStatus.Off;
                    break;
                case DieselEngineState.Running:
                    Indicators[6] = IndicatorStatus.On;
                    Indicators[7] = IndicatorStatus.Off;
                    break;
                case DieselEngineState.Stopping:
                    Indicators[6] = IndicatorStatus.Off;
                    Indicators[7] = IndicatorStatus.Flash;
                    break;
            }
            bool PowerAvailable = false;
            bool DynamicBrakeAvailable = true;
            if (CurrentPowerMode == PowerMode.AC25kV)
            {
                if (CurrentPantographState() == PantographState.Up)
                {
                    if (PantographVoltageV() < 4250 && PantographVoltageV() > 1900 && (CurrentTractionSystemChangeState == TractionSystemChangeState.Inactive || CurrentTractionSystemChangeState == TractionSystemChangeState.AuxiliaryFunction) && SimulatedLineVoltageV() < 4250 && SimulatedLineVoltageV() > 1900)
                    {
                        PantoUp = false;
                        TargetPowerMode = PowerMode.DC3kV;
                    }
                }

                PowerAvailable = CurrentCircuitBreakerState() == CircuitBreakerState.Closed;
                if (PantographVoltageV() < 17500 || PantographVoltageV() > 30000)
                {
                    PowerAvailable = false;
                }
                else if (PantographVoltageV() < 22500)
                {
                    MaximumPowerW = (PantographVoltageV()-17500)*2.16e6f/5000;
                }
                else if (PantographVoltageV() < 25000)
                {
                    MaximumPowerW = 2.16e6f+(PantographVoltageV()-22500)*0.24e6f/2500;
                }
                else if (PantographVoltageV() > 29000)
                {
                    MaximumPowerW = 2.4e6f*(1-(PantographVoltageV()-29000)/1000.0f);
                }
                else
                {
                    MaximumPowerW = 2.4e6f;
                }
                if (IsMultipleUnit && PantographVoltageV() > 22500) MaximumPowerW = Math.Min(MaximumPowerW, 2.3e6f*0.96f*PantographVoltageV()/22500);
                if (CurrentCircuitBreakerState() != CircuitBreakerState.Closed)
                {
                    MaximumDynamicBrakePowerW = 2.0e6f;
                }
                else if (PantographVoltageV() < 17500 || PantographVoltageV() > 31000)
                {
                    DynamicBrakeAvailable = false;
                }
                else if (PantographVoltageV() < 22500)
                {
                    MaximumDynamicBrakePowerW = (PantographVoltageV()-17500)*2.16e6f/5000;
                }
                else if (PantographVoltageV() < 25000)
                {
                    MaximumDynamicBrakePowerW = 2.16e6f+(PantographVoltageV()-22500)*0.24e6f/2500;
                }
                else if (PantographVoltageV() > 29000)
                {
                    MaximumDynamicBrakePowerW = 2.0e6f;
                }
                else if (PantographVoltageV() > 30000)
                {
                    MaximumDynamicBrakePowerW = 2.0e6f*(1-(PantographVoltageV()-30000)/1000.0f);
                }
                else
                {
                    MaximumDynamicBrakePowerW = 2.4e6f;
                }
                SignalEventToCircuitBreaker(PowerSupplyEvent.CloseCircuitBreaker, (int)RemoteCircuitBreakerState+((int)CurrentPowerMode)*4+(LocalPantoUp ? 16 : 0));
                AvailableTractionPowerW = float.MaxValue;
            }
            else if (CurrentPowerMode == PowerMode.DC3kV)
            {
                if (CurrentPantographState() == PantographState.Up)
                {
                    if (PantographVoltageV() > 16000 && (CurrentTractionSystemChangeState == TractionSystemChangeState.Inactive || CurrentTractionSystemChangeState == TractionSystemChangeState.AuxiliaryFunction) && SimulatedLineVoltageV() > 16000)
                    {
                        PantoUp = false;
                        TargetPowerMode = PowerMode.AC25kV;
                    }
                }

                PowerAvailable = CurrentCircuitBreakerState() == CircuitBreakerState.Closed;
                if (PantographVoltageV() < 2000 || PantographVoltageV() > 4100)
                {
                    PowerAvailable = false;
                }
                else if (PantographVoltageV() < 2700)
                {
                    MaximumPowerW = (PantographVoltageV()-2000)*1.8e6f/700;
                }
                else if (PantographVoltageV() < 3000)
                {
                    MaximumPowerW = 1.8e6f+(PantographVoltageV()-2700)*0.2e6f/300;
                }
                else if (PantographVoltageV() > 4000)
                {
                    MaximumPowerW = 2.0e6f*(1-(PantographVoltageV()-4000)/100.0f);
                }
                else
                {
                    MaximumPowerW = 2.0e6f;
                }
                if (IsMultipleUnit && PantographVoltageV() > 2700) MaximumPowerW = Math.Min(MaximumPowerW, 2.0e6f*0.7f*PantographVoltageV()/2700);
                if (CurrentCircuitBreakerState() != CircuitBreakerState.Closed)
                {
                    MaximumDynamicBrakePowerW = 2.0e6f;
                }
                else if (PantographVoltageV() < 2000 || PantographVoltageV() > 4100)
                {
                    DynamicBrakeAvailable = false;
                }
                else if (PantographVoltageV() < 2700)
                {
                    MaximumDynamicBrakePowerW = (PantographVoltageV()-2000)*1.8e6f/700;
                }
                else if (PantographVoltageV() < 3000)
                {
                    MaximumDynamicBrakePowerW = 1.8e6f+(PantographVoltageV()-2700)*0.2e6f/300;
                }
                else if (PantographVoltageV() > 4000)
                {
                    MaximumDynamicBrakePowerW = 2.0e6f*(1-(PantographVoltageV()-4000)/100.0f);
                }
                else
                {
                    MaximumDynamicBrakePowerW = 2.0e6f;
                }
                SignalEventToCircuitBreaker(PowerSupplyEvent.CloseCircuitBreaker, (int)RemoteCircuitBreakerState+((int)CurrentPowerMode)*4+(LocalPantoUp ? 16 : 0));
                AvailableTractionPowerW = float.MaxValue;
            }
            else if (CurrentPowerMode == PowerMode.Diesel)
            {
                AvailableTractionPowerW = Math.Max(LocalCETPowerW-3.0e5f, 0)*0.95f;
                MaximumPowerW = Math.Max(1.2e6f, Math.Min(AvailableTractionPowerW, 1.4e6f));
                MaximumDynamicBrakePowerW = 2.0e6f;
                if (RemoteCETCircuitBreakerState > LocalCETCircuitBreakerState)
                {
                    PowerAvailable = false;
                    SignalEventToCircuitBreaker(PowerSupplyEvent.CloseCircuitBreaker, (int)RemoteCETCircuitBreakerState+((int)CurrentPowerMode)*4);
                }
                else
                {
                    PowerAvailable = CurrentCircuitBreakerState() == CircuitBreakerState.Closed;
                    SignalEventToCircuitBreaker(PowerSupplyEvent.CloseCircuitBreaker, (int)LocalCETCircuitBreakerState+((int)CurrentPowerMode)*4);
                }
                DynamicBrakeAvailable &= CurrentCircuitBreakerState() == CircuitBreakerState.Closed;
            }

            bool roof = false;
            bool train = false;
            if (CurrentPowerMode == PowerMode.Diesel)
            {
                if ((CurrentTractionSystemChangeState == TractionSystemChangeState.Inactive || CurrentTractionSystemChangeState >= TractionSystemChangeState.ReconfigureLower)
                    || (CurrentTractionSystemChangeState == TractionSystemChangeState.AuxiliaryFunction && LocalCabActive)
                    || (IsRear && CurrentTractionSystemChangeState <= TractionSystemChangeState.WaitNeutral)
                    || (!IsRear && CurrentTractionSystemChangeState >= TractionSystemChangeState.WaitEnd)
                )
                {
                    train = true;
                }
            }
            else if ((CurrentTractionSystemChangeState == TractionSystemChangeState.Inactive || CurrentTractionSystemChangeState >= TractionSystemChangeState.ReconfigureLower) && (IsRear || IsMultipleUnit || PantographSelection != PantographSelector.Both))
            {
                roof = CurrentPowerMode == PowerMode.AC25kV && PantoUp;
                train = true;
            }
            else if (CurrentTractionSystemChangeState != TractionSystemChangeState.Inactive && TractionSystemChangeDefrost)
            {
                if (CurrentTractionSystemChangeState == TractionSystemChangeState.AuxiliaryFunction)
                {
                    train = LocalCabActive;
                }
                else if ((IsRear && CurrentTractionSystemChangeState < TractionSystemChangeState.WaitNeutral)
                    || (!IsRear && CurrentTractionSystemChangeState >= TractionSystemChangeState.WaitEnd)
                )
                {
                    train = true;
                }
            }
            if (roof) LocalRoofLineContactorState = CircuitBreakerState.Closed;
            else LocalRoofLineContactorState = CircuitBreakerState.Open;
            if (train) LocalTrainLineContactorState = CircuitBreakerState.Closed;
            else LocalTrainLineContactorState = CircuitBreakerState.Open;

            SignalEventToOtherLocomotive(RemoteHeadIndex, PowerSupplyEvent.CloseCircuitBreaker, (int)LocalCircuitBreakerState+((int)LocalTrainLineContactorState)*4+((int)LocalRoofLineContactorState)*16);
            SignalEventToOtherLocomotive(RemoteHeadIndex, PowerSupplyEvent.RaisePantograph, (int)LocalPantographVoltageV);
            PowerAvailable &= (!ParkingMode || !TargetParkingMode);
            DynamicBrakeAvailable &= PowerAvailable || CurrentTractionSystemChangeState == TractionSystemChangeState.Inactive;
            SetCurrentDynamicBrakeAvailability(DynamicBrakeAvailable);
            if (LocalCabActive)
            {
                LeadIndex = IndexOfLocomotive();
                SignalEventToOtherTrainVehiclesWithId(PowerSupplyEvent.TurnOnMasterKey, LeadIndex);
                SignalEventToOtherLocomotivesWithId(PowerSupplyEvent.IncreaseVoltageSelectorPosition, (int)CurrentPowerMode+((int)TargetPowerMode)*4);
                if (PantographSelectorPosition != null)
                {
                    if (PantographSelectorPosition.Name == "1") PantographSelection = PantographSelector.Front;
                    else if (PantographSelectorPosition.Name == "2") PantographSelection = PantographSelector.Rear;
                    else if (PantographSelectorPosition.Name == "1+2") PantographSelection = PantographSelector.Both;
                    else PantographSelection = PantographSelector.Automatic;
                }
                SignalEventToOtherLocomotivesWithId(PowerSupplyEvent.IncreasePantographSelectorPosition, (int)PantographSelection);
            }
            else if (IsLocomotiveLeading && !RemoteCabActive)
            {
                LeadIndex = -1;
                SignalEventToOtherTrainVehiclesWithId(PowerSupplyEvent.TurnOnMasterKey, LeadIndex);
            }
            if (!LocalCabActive && MasterKeyOn()) SignalEventToMasterKey(PowerSupplyEvent.TurnOffMasterKey);
            UpdatePantographs();
            if (!LocalCabActive && !RemoteCabActive)
            {
                if (PantoUp && !QuickPowerOn && !ParkingMode) PantoUp = false;
            }
            if (ParkingMode && TargetParkingMode) Indicators[1000] = IndicatorStatus.On;
            else if (!ParkingMode && TargetParkingMode) Indicators[1000] = IndicatorStatus.Flash;
            else Indicators[1000] = IndicatorStatus.Off;
            Indicators[1001] = ParkingMode && !TargetParkingMode ? IndicatorStatus.Flash : IndicatorStatus.Off;
            if (TargetParkingMode && !ParkingMode)
            {
                ParkingMode = true;
            }
            else if (ParkingMode && !TargetParkingMode)
            {
                if (CurrentMainPowerSupplyState() == PowerSupplyState.PowerOn) ParkingMode = false;
            }
            if (ParkingMode)
            {
                if (CurrentPowerMode == PowerMode.Diesel)
                {
                    if (LocalCETGeneratorState == PowerSupplyState.PowerOff) ParkingMode = TargetParkingMode = false;
                }
                else
                {
                    if (!PantoUp) ParkingMode = TargetParkingMode = false;
                    else if (LocalCabActive || RemoteCabActive)
                    {
                        int prevTarget = PantoTarget;
                        HandlePantographs();
                        if (prevTarget == PantoTarget && CircuitBreakerClosingAuthorization() && LocalCircuitBreakerState == CircuitBreakerState.Open)
                        {
                            SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOn);
                        }
                    }
                }

            }
            if (QuickPowerOn)
            {
                ParkingMode = false;
                if (IsLocomotiveLeading && !LocalCabActive)
                {
                    LocalCabActive = true;
                    RemoteCabActive = false;
                    LeadIndex = -1;
                    if (CurrentPowerMode != PowerMode.Diesel) PantoUp = true;
                    SignalEventToMasterKey(PowerSupplyEvent.TurnOnMasterKey);
                }
                if (CurrentPowerMode == PowerMode.Diesel)
                {
                    if (LocalCabActive || RemoteCabActive) QuickPowerOn = false;
                }
                else
                {
                    if (CurrentPantographState() != PantographState.Up) HandlePantographs();
                    if (!PantoUp || CurrentCircuitBreakerState() == CircuitBreakerState.Closed)
                    {
                        QuickPowerOn = false;
                    }
                    else if (LocalPantoUp)
                    {
                        if (CircuitBreakerClosingAuthorization())
                        {
                            QuickPowerOn = false;
                            SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOn);
                        }
                    }
                }
            }
            switch (CurrentMainPowerSupplyState())
            {
                case PowerSupplyState.PowerOn:
                    if (!PowerAvailable)
                    {
                        PowerOnTimer.Stop();
                        SetCurrentMainPowerSupplyState(PowerSupplyState.PowerOff);
                        SignalEvent(Event.EnginePowerOff);
                    }
                    break;
                case PowerSupplyState.PowerOnOngoing:
                    if (PowerOnTimer.Triggered)
                    {
                        SetCurrentMainPowerSupplyState(PowerSupplyState.PowerOn);
                        SignalEvent(Event.EnginePowerOn);
                    }
                    if (!PowerAvailable)
                    {
                        PowerOnTimer.Stop();
                        SetCurrentMainPowerSupplyState(PowerSupplyState.PowerOff);
                    }
                    break;
                case PowerSupplyState.PowerOff:
                    if (PowerAvailable)
                    {
                        PowerOnTimer.Start();
                        SetCurrentMainPowerSupplyState(PowerSupplyState.PowerOnOngoing);
                    }
                    break;
            }
            bool AuxiliaryPowerAvailable = CurrentCircuitBreakerState() == CircuitBreakerState.Closed;
            if (!AuxiliaryPowerAvailable && SpeedMpS() > MpS.FromKpH(60) && CurrentDynamicBrakeAvailability())
                PowerSupplyDynamicBrakePercent = 10;
            else
                PowerSupplyDynamicBrakePercent = -1;
            bool SupplyBrakeActive = ThrottlePercent() == 0 && SpeedMpS() > MpS.FromKpH(15) && CurrentDynamicBrakeAvailability() && PowerSupplyDynamicBrakePercent > 9 && CurrentPantographState() == PantographState.Up;
            AuxiliaryPowerAvailable |= SupplyBrakeActive;
            switch (CurrentAuxiliaryPowerSupplyState())
            {
                case PowerSupplyState.PowerOn:
                    if (!AuxiliaryPowerAvailable)
                    {
                        AuxPowerOnTimer.Stop();
                        SetCurrentAuxiliaryPowerSupplyState(PowerSupplyState.PowerOff);
                        SignalEvent(Event.PowerConverterOff);
                    }
                    break;
                case PowerSupplyState.PowerOnOngoing:
                    if (AuxPowerOnTimer.Triggered)
                    {
                        SetCurrentAuxiliaryPowerSupplyState(PowerSupplyState.PowerOn);
                        SignalEvent(Event.PowerConverterOn);
                    }
                    if (!AuxiliaryPowerAvailable)
                    {
                        AuxPowerOnTimer.Stop();
                        SetCurrentAuxiliaryPowerSupplyState(PowerSupplyState.PowerOff);
                    }
                    break;
                case PowerSupplyState.PowerOff:
                    if (AuxiliaryPowerAvailable)
                    {
                        AuxPowerOnTimer.Start();
                        SetCurrentAuxiliaryPowerSupplyState(PowerSupplyState.PowerOnOngoing);
                    }
                    break;
            }
            
            bool ElectricSupplyAvailable = LocalTrainLineContactorState == CircuitBreakerState.Closed;
            switch (CurrentPowerMode)
            {
                case PowerMode.DC3kV:
                case PowerMode.AC25kV:
                    ElectricSupplyAvailable &= SupplyBrakeActive || LocalCircuitBreakerState == CircuitBreakerState.Closed;
                    break;
                case PowerMode.Diesel:
                    ElectricSupplyAvailable = false;
                    break;
            }
            SetCurrentElectricTrainSupplyState(ElectricSupplyAvailable ? PowerSupplyState.PowerOn : PowerSupplyState.PowerOff);

            SetCurrentBatteryState(BatterySwitchOn() ? PowerSupplyState.PowerOn : PowerSupplyState.PowerOff);
            SetCurrentLowVoltagePowerSupplyState(BatterySwitchOn() ? PowerSupplyState.PowerOn : PowerSupplyState.PowerOff);
            SetCurrentCabPowerSupplyState(BatterySwitchOn() && MasterKeyOn() ? PowerSupplyState.PowerOn : PowerSupplyState.PowerOff);

            if (elapsedClockSeconds > 0)
                LocalPantographVoltageV = PantographFilter.Filter(LocalPantoUp ? SimulatedLineVoltageV() : 0.0f, elapsedClockSeconds);
            if (!LocalPantoUp && CurrentPantographState() == PantographState.Up) SetPantographVoltageV(RemotePantographVoltageV);
            else SetPantographVoltageV(LocalPantographVoltageV);
            PantographVoltageVAC = CurrentPowerMode == PowerMode.AC25kV ? PantographVoltageV() : float.NaN;
            PantographVoltageVDC = CurrentPowerMode != PowerMode.AC25kV ? PantographVoltageV() : float.NaN;
            if (CurrentPowerMode == PowerMode.Diesel)
            {
                SetFilterVoltageV(VoltageFilter.Filter(CurrentCircuitBreakerState() == CircuitBreakerState.Closed ? 3000 : 0, elapsedClockSeconds));
            }
            else
            {
                SetFilterVoltageV(VoltageFilter.Filter(CurrentCircuitBreakerState() == CircuitBreakerState.Closed ? PantographVoltageV() : 0.0f, elapsedClockSeconds));
            }
            UpdateIndicators();
        }
        public void UpdateIndicators()
        {
            foreach (var kvp in Indicators)
            {
                int estado = 0;
                if (kvp.Value == IndicatorStatus.Off) estado = 0;
                else if (kvp.Value == IndicatorStatus.On) estado = 1;
                else if ((kvp.Value == IndicatorStatus.CounterFlash) ^ Blink.On) estado = 1;
                if (kvp.Key == 1000) ServiceRetentionButton = estado == 1;
                else if (kvp.Key == 1001) ServiceRetentionCancellationButton = estado == 1;
                else SetCabDisplayControl(kvp.Key, estado);
            }
        }

        public void HandlePantographs()
        {
            if (CurrentPowerMode == PowerMode.Diesel || !PantoUp)
            {
                PantoTarget = 0;
            }
            else if (CurrentPantographDownSectionState >= PantographDownSectionState.LowerPantograph && CurrentPantographDownSectionState < PantographDownSectionState.RaisePantograph)
            {
                PantoTarget = 0;
            }
            else if (CurrentTractionSystemChangeState == TractionSystemChangeState.AuxiliaryFunction)
            {
                if (LocalCabActive) PantoTarget = TargetPowerMode == PowerMode.DC3kV ? 2 : 1;
            }
            else if (CurrentTractionSystemChangeState == TractionSystemChangeState.ReconfigureLower)
            {
                PantoTarget = 0;
            }
            else if (CurrentTractionSystemChangeState != TractionSystemChangeState.Inactive && CurrentTractionSystemChangeState != TractionSystemChangeState.ReconfigureRaise)
            {
                if (IsRear)
                {
                    if (CurrentTractionSystemChangeState >= TractionSystemChangeState.LowerOldPantograph) PantoTarget = 0;
                    else PantoTarget = CurrentPowerMode == PowerMode.DC3kV ? 2 : 1;
                }
                else
                {
                    if (CurrentTractionSystemChangeState >= TractionSystemChangeState.RaiseNewPantograph) PantoTarget = TargetPowerMode == PowerMode.DC3kV ? 2 : 1;
                    else PantoTarget = 0;
                }
            }
            else if (PantographSelection == PantographSelector.Both)
            {
                if (CurrentPowerMode == PowerMode.AC25kV) PantoTarget = 0;
                else if (IsMultipleUnit && (IsRear != IsRearUnit))
                {
                    PantoTarget = CurrentPowerMode == PowerMode.DC3kV ? 3 : 4;
                }
                else
                {
                    PantoTarget = CurrentPowerMode == PowerMode.DC3kV ? 2 : 1;
                }
            }
            else if ((PantographSelection != PantographSelector.Front) == IsRear)
            {
                PantoTarget = CurrentPowerMode == PowerMode.DC3kV ? 2 : 1;
            }
            else if (RemoteHeadIndex >= 0)
            {
                PantoTarget = CurrentPowerMode == PowerMode.DC3kV ? 3 : 4;
            }
            else if (PantographSelection == PantographSelector.Automatic)
            {
                PantoTarget = CurrentPowerMode == PowerMode.DC3kV ? 2 : 1;
            }
        }
        public void UpdatePantographs()
        {
            if (PantographSelection == PantographSelector.Both && CurrentPowerMode == PowerMode.AC25kV) PantoUp = false;
            if (RemoteHeadIndex < 0 && PantoTarget > 2) PantoUp = false;
            int panto = PantoTarget;
            if (Flipped && panto == 1) panto = 2;
            else if (Flipped && panto == 2) panto = 1;
            LocalPantoUp = CurrentPantographState(1) == PantographState.Up || CurrentPantographState(2) == PantographState.Up;
            if (LocalCircuitBreakerState == CircuitBreakerState.Closed && panto != 1 && panto != 2)
            {
                SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
            }
            for (int i=1; i<5; i++)
            {
                var state = CurrentPantographState(i);
                if (panto == i)
                {
                    if (state == PantographState.Down || state == PantographState.Lowering) SignalEventToPantograph(PowerSupplyEvent.RaisePantograph, i);
                }
                else
                {
                    if ((state == PantographState.Up || state == PantographState.Raising) && LocalCircuitBreakerState == CircuitBreakerState.Open) SignalEventToPantograph(PowerSupplyEvent.LowerPantograph, i);
                }
            }
        }
        
        public override void HandleEvent(PowerSupplyEvent evt)
        {
            switch (evt)
            {
                case PowerSupplyEvent.TurnOnMasterKey:
                case PowerSupplyEvent.TurnOffMasterKey:
                    if (RemoteCabActive) return;
                    LocalCabActive = evt == PowerSupplyEvent.TurnOnMasterKey;
                    SignalEventToMasterKey(evt);
                    if (!LocalCabActive) QuickPowerOn = false;
                    break;
                case PowerSupplyEvent.OpenCircuitBreakerButtonReleased:
                    if (!LocalCabActive) return;
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        SignalEventToOtherTrainVehicles(PowerSupplyEvent.OpenTractionCutOffRelayButtonReleased);
                    }
                    else
                    {
                        SignalEventToCircuitBreaker(evt);
                        SignalEventToOtherTrainVehicles(evt);
                    }
                    break;
                case PowerSupplyEvent.OpenCircuitBreakerButtonPressed:
                    if (!LocalCabActive) return;
                    CurrentNeutralSectionState = NeutralSectionState.Inactive;
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        SignalEventToOtherTrainVehicles(PowerSupplyEvent.OpenTractionCutOffRelayButtonPressed);
                    }
                    else
                    {
                        SignalEventToCircuitBreaker(evt);
                        SignalEventToOtherTrainVehicles(evt);
                    }
                    break;
                case PowerSupplyEvent.CloseCircuitBreakerButtonReleased:
                    if (!LocalCabActive) return;
                    QuickPowerOn = false;
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        SignalEventToOtherTrainVehicles(PowerSupplyEvent.CloseTractionCutOffRelayButtonReleased);
                    }
                    else
                    {
                        SignalEventToCircuitBreaker(evt);
                        SignalEventToOtherTrainVehicles(evt);
                    }
                    break;
                case PowerSupplyEvent.CloseCircuitBreakerButtonPressed:
                    if (!LocalCabActive) return;
                    if (CurrentPowerMode == PowerMode.Diesel)
                    {
                        SignalEventToOtherTrainVehicles(PowerSupplyEvent.CloseTractionCutOffRelayButtonPressed);
                    }
                    else if (CurrentNeutralSectionState == NeutralSectionState.Inactive)
                    {
                        SignalEventToCircuitBreaker(evt);
                        SignalEventToOtherTrainVehicles(evt);
                    }
                    break;
                case PowerSupplyEvent.OpenCircuitBreaker:
                case PowerSupplyEvent.CloseCircuitBreaker:
                    break;
                case PowerSupplyEvent.LowerPantograph:
                case PowerSupplyEvent.RaisePantograph:
                    if (!LocalCabActive) return;
                    PantoUp = !PantoUp;
                    SignalEventToOtherLocomotives(PantoUp ? PowerSupplyEvent.RaisePantograph : PowerSupplyEvent.LowerPantograph);
                    break;
                case PowerSupplyEvent.QuickPowerOn:
                    QuickPowerOn = true;
                    PantoUp = true;
                    if (SimulatedLineVoltageV() > 10000) CurrentPowerMode = TargetPowerMode = PowerMode.AC25kV;
                    else if (SimulatedLineVoltageV() > 1800) CurrentPowerMode = TargetPowerMode = PowerMode.DC3kV;
                    else CurrentPowerMode = TargetPowerMode = PowerMode.Diesel;
                    SignalEventToBatterySwitch(PowerSupplyEvent.QuickPowerOn);
                    SignalEventToElectricTrainSupplySwitch(PowerSupplyEvent.SwitchOnElectricTrainSupply);
                    break;

                case PowerSupplyEvent.QuickPowerOff:
                    QuickPowerOn = false;
                    PantoUp = false;
                    SignalEventToElectricTrainSupplySwitch(PowerSupplyEvent.SwitchOffElectricTrainSupply);
                    SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
                    SignalEventToMasterKey(PowerSupplyEvent.TurnOffMasterKey);
                    SignalEventToBatterySwitch(PowerSupplyEvent.QuickPowerOff);
                    break;
                case PowerSupplyEvent.IncreaseVoltageSelectorPosition:
                case PowerSupplyEvent.DecreaseVoltageSelectorPosition:
                    break;
                case PowerSupplyEvent.ServiceRetentionButtonPressed:
                case PowerSupplyEvent.ServiceRetentionCancellationButtonPressed:
                    if (LocalCabActive)
                    {
                        bool act = evt == PowerSupplyEvent.ServiceRetentionButtonPressed;
                        if (TargetParkingMode != act)
                        {
                            TargetParkingMode = act;
                            SignalEventToOtherLocomotives(evt);
                        }
                    }
                    break;
                default:
                    base.HandleEvent(evt);
                    break;
            }
        }
        
        public override void HandleEvent(PowerSupplyEvent evt, int id)
        {
            switch(evt)
            {
                case PowerSupplyEvent.LowerPantograph:
                case PowerSupplyEvent.RaisePantograph:
                    if (id == 1)
                    {
                        HandleEvent(evt);
                    }
                    break;
                case PowerSupplyEvent.GenericPowerSupplyButtonPressed:
                    switch (id)
                    {
                        case 0:
                            if (CurrentTractionSystemChangeState == TractionSystemChangeState.Inactive)
                            {
                                CurrentTractionSystemChangeState = TractionSystemChangeState.Preparation;
                                SignalEventToOtherTrainVehiclesWithId(evt, id);
                            }
                            else if (CurrentTractionSystemChangeState < TractionSystemChangeState.ReconfigureLower)
                            {
                                if (CurrentTractionSystemChangeState > TractionSystemChangeState.Prepared && CurrentPowerMode == PowerMode.Diesel)
                                    CurrentPowerMode = TargetPowerMode;
                                CurrentTractionSystemChangeState = TractionSystemChangeState.ReconfigureLower;
                                SignalEventToOtherTrainVehiclesWithId(evt, id);
                            }
                            break;
                        case 1:
                            if (CurrentTractionSystemChangeState == TractionSystemChangeState.Preparation || CurrentTractionSystemChangeState == TractionSystemChangeState.Prepared)
                            {
                                LocomotivePositionDelay.Setup(CurrentPowerMode == PowerMode.Diesel ? 80 : 30);
                                LocomotivePositionDelay.Start();
                                CurrentTractionSystemChangeState = TractionSystemChangeState.WaitNeutral;
                                if (CurrentPowerMode != PowerMode.Diesel) TargetPowerMode = CurrentPowerMode == PowerMode.DC3kV ? PowerMode.AC25kV : PowerMode.DC3kV;
                                SignalEventToOtherTrainVehiclesWithId(evt, id);
                            }
                            break;
                        case 2:
                            if (CurrentTractionSystemChangeState > TractionSystemChangeState.Prepared && CurrentTractionSystemChangeState != TractionSystemChangeState.AuxiliaryFunction) return;
                            if (CurrentPowerMode == PowerMode.Diesel || (CurrentPowerMode != TargetPowerMode && TargetPowerMode == PowerMode.Diesel))
                            {
                                TargetPowerMode = PowerMode.DC3kV;
                            }
                            else if (CurrentPowerMode != PowerMode.Diesel || (CurrentPowerMode != TargetPowerMode && TargetPowerMode != PowerMode.Diesel))
                            {
                                TargetPowerMode = PowerMode.Diesel;
                            }
                            break;
                        case 3:
                            if (CurrentNeutralSectionState != NeutralSectionState.Inactive)
                            {
                                CurrentNeutralSectionState = NeutralSectionState.Inactive;
                                SignalCustomEvent(CustomEvent.EndNeutralSection);
                            }
                            else if (CurrentPantographDownSectionState != PantographDownSectionState.Inactive && CurrentPantographDownSectionState != PantographDownSectionState.RaisePantograph)
                            {
                                PantoUp = false;
                                SignalEventToOtherTrainVehicles(PowerSupplyEvent.LowerPantograph);
                            }
                            else
                            {
                                LocomotivePositionDelay.Setup(SpeedMpS() * 6);
                                LocomotivePositionDelay.Start();
                                CurrentNeutralSectionState = NeutralSectionState.Preparation;
                                NeutralSectionByTCS = false;
                                SignalCustomEvent(CustomEvent.StartNeutralSection);
                            }
                            break;
                        case 4:
                            if (!PantoUp && TargetPowerMode == PowerMode.DC3kV) TargetPowerMode = PowerMode.AC25kV;
                            else if (!PantoUp && TargetPowerMode == PowerMode.AC25kV) SimulatedVoltage = 25000;
                            SignalEventToOtherLocomotivesWithId(evt, id);
                            break;
                        case 5:
                            if (!PantoUp && TargetPowerMode == PowerMode.AC25kV) TargetPowerMode = PowerMode.DC3kV;
                            else if (!PantoUp && TargetPowerMode == PowerMode.DC3kV) SimulatedVoltage = 3000;
                            SignalEventToOtherLocomotivesWithId(evt, id);
                            break;
                        case 6:
                            /*SignalEventToOtherLocomotives(PowerSupplyEvent.StartEngine);
                            Confirm(CabControl.HelperDiesel, CabSetting.On);*/
                            SignalEventToHelperEngines(PowerSupplyEvent.StartEngine);
                            break;
                        case 7:
                             /*SignalEventToOtherLocomotives(PowerSupplyEvent.StopEngine);
                             Confirm(CabControl.HelperDiesel, CabSetting.Off);*/
                             SignalEventToHelperEngines(PowerSupplyEvent.StopEngine);
                            break;
                        case 8:
                            if (CurrentTractionSystemChangeState == TractionSystemChangeState.Preparation || CurrentTractionSystemChangeState == TractionSystemChangeState.Prepared)
                            {
                                CurrentTractionSystemChangeState = TractionSystemChangeState.AuxiliaryFunction;
                                SignalEventToOtherTrainVehiclesWithId(evt, id);
                            }
                            break;
                    }
                    break;
                case PowerSupplyEvent.CloseCircuitBreaker:
                    LocalCircuitBreakerState = (CircuitBreakerState)id;
                    break;
                case PowerSupplyEvent.GenericPowerSupplySwitchOff:
                case PowerSupplyEvent.GenericPowerSupplySwitchOn:
                    break;
                default:
                    base.HandleEvent(evt, id);
                    break;
            }
        }
        
        public override void HandleEventFromLeadLocomotive(PowerSupplyEvent evt)
        {
            switch (evt)
            {
                case PowerSupplyEvent.OpenCircuitBreakerButtonPressed:
                    CurrentNeutralSectionState = NeutralSectionState.Inactive;
                    QuickPowerOn = false;
                    SignalEventToCircuitBreaker(evt);
                    break;
                case PowerSupplyEvent.CloseCircuitBreakerButtonPressed:
                    CurrentNeutralSectionState = NeutralSectionState.Inactive;
                    SignalEventToCircuitBreaker(evt);
                    break;
                case PowerSupplyEvent.ServiceRetentionButtonPressed:
                    TargetParkingMode = true;
                    break;
                case PowerSupplyEvent.ServiceRetentionCancellationButtonPressed:
                    TargetParkingMode = false;
                    break;
                case PowerSupplyEvent.QuickPowerOn:
                    QuickPowerOn = true;
                    if (SimulatedLineVoltageV() > 10000) CurrentPowerMode = TargetPowerMode = PowerMode.AC25kV;
                    else if (SimulatedLineVoltageV() > 1800) CurrentPowerMode = TargetPowerMode = PowerMode.DC3kV;
                    else CurrentPowerMode = TargetPowerMode = PowerMode.Diesel;
                    SignalEventToBatterySwitch(PowerSupplyEvent.QuickPowerOn);
                    SignalEventToElectricTrainSupplySwitch(PowerSupplyEvent.SwitchOnElectricTrainSupply);
                    break;
                case PowerSupplyEvent.QuickPowerOff:
                    QuickPowerOn = false;
                    SignalEventToElectricTrainSupplySwitch(PowerSupplyEvent.SwitchOffElectricTrainSupply);
                    SignalEventToCircuitBreaker(PowerSupplyEvent.QuickPowerOff);
                    SignalEventToBatterySwitch(PowerSupplyEvent.QuickPowerOff);
                    break;
                case PowerSupplyEvent.RaisePantograph:
                    PantoUp = true;
                    break;
                case PowerSupplyEvent.LowerPantograph:
                    PantoUp = false;
                    break;
                default:
                    base.HandleEventFromLeadLocomotive(evt);
                    break;
            }
        }
        public void SignalCustomEvent(CustomEvent evt)
        {
            SignalEventToOtherLocomotivesWithId(PowerSupplyEvent.CloseCircuitBreakerButtonPressed, (int)evt);
        }
        public void HandleCustomEvent(CustomEvent evt)
        {
            switch (evt)
            {
                case CustomEvent.StartNeutralSection:
                case CustomEvent.StartNeutralSectionTCS:
                    {
                        NeutralSectionByTCS = evt == CustomEvent.StartNeutralSectionTCS;
                        float delay = SpeedMpS() * 6;
                        if (IsRear && !IsRearUnit) delay += 158;
                        else if (!IsRear && IsRearUnit) delay += 188;
                        else if (IsRear && IsRearUnit) delay += 346;
                        LocomotivePositionDelay.Setup(delay);
                        LocomotivePositionDelay.Start();
                        CurrentNeutralSectionState = NeutralSectionState.Preparation;
                    }
                    break;
                case CustomEvent.EndNeutralSection:
                    CurrentNeutralSectionState = NeutralSectionState.Inactive;
                    break;
                case CustomEvent.EndNeutralSectionTCS:
                    {
                        float delay = 0;
                        if (IsRear && !IsRearUnit) delay += 188;
                        else if (!IsRear && IsRearUnit) delay += 218;
                        else if (IsRear && IsRearUnit) delay += 376;
                        LocomotivePositionDelay.Setup(delay);
                        LocomotivePositionDelay.Start();
                    }
                    break;
                case CustomEvent.LowerPantographTCS:
                    {
                        float delay = 0;
                        if (IsRear && !IsRearUnit) delay = 158;
                        else if (!IsRear && IsRearUnit) delay = 188;
                        else if (IsRear && IsRearUnit) delay = 346;
                        LocomotivePositionDelay.Setup(delay);
                        LocomotivePositionDelay.Start();
                        CurrentPantographDownSectionState = PantographDownSectionState.Preparation;
                    }
                    break;
                case CustomEvent.RaisePantographTCS:
                    if (CurrentPantographDownSectionState == PantographDownSectionState.Active)
                    {
                        float delay = 0;
                        if (IsRear && !IsRearUnit) delay = 188;
                        else if (!IsRear && IsRearUnit) delay = 218;
                        else if (IsRear && IsRearUnit) delay = 376;
                        LocomotivePositionDelay.Setup(delay);
                        LocomotivePositionDelay.Start();
                    }
                    else if (CurrentPantographDownSectionState != PantographDownSectionState.Inactive && CurrentPantographDownSectionState < PantographDownSectionState.Active)
                    {
                        CurrentPantographDownSectionState = PantographDownSectionState.Active;
                        float delay = 0;
                        if (IsRear && !IsRearUnit) delay = 188;
                        else if (!IsRear && IsRearUnit) delay = 218;
                        else if (IsRear && IsRearUnit) delay = 376;
                        LocomotivePositionDelay.Setup(delay);
                        LocomotivePositionDelay.Start();
                    }
                    break;
            }
        }
        public override void HandleEventFromLeadLocomotive(PowerSupplyEvent evt, int id)
        {
            switch (evt)
            {
                case PowerSupplyEvent.TurnOnMasterKey:
                    LeadIndex = id;
                    RemoteCabActive = id >= 0;
                    if (LocalCabActive && RemoteCabActive)
                    {
                        LocalCabActive = false;
                        SignalEventToMasterKey(PowerSupplyEvent.TurnOffMasterKey);
                    }
                    break;
                case PowerSupplyEvent.CloseCircuitBreakerButtonPressed:
                    HandleCustomEvent((CustomEvent)id);
                    break;
                case PowerSupplyEvent.IncreaseVoltageSelectorPosition:
                    {
                        var current = (PowerMode)(id&3);
                        id /= 4;
                        var target = (PowerMode)(id&3);
                        TargetPowerMode = target;
                    }
                    break;
                case PowerSupplyEvent.IncreasePantographSelectorPosition:
                    PantographSelection = (PantographSelector)id;
                    break;
                case PowerSupplyEvent.GenericPowerSupplyButtonPressed:
                    switch (id)
                    {
                        case 0:
                            if (CurrentTractionSystemChangeState == TractionSystemChangeState.Inactive)
                                CurrentTractionSystemChangeState = TractionSystemChangeState.Preparation;
                            else
                            {
                                if (CurrentTractionSystemChangeState > TractionSystemChangeState.Prepared && CurrentPowerMode != PowerMode.Diesel)
                                    CurrentPowerMode = TargetPowerMode;
                                CurrentTractionSystemChangeState = TractionSystemChangeState.ReconfigureLower;
                            }
                            break;
                        case 1:
                            {
                                float delay = 0;
                                if (IsRear && !IsRearUnit) delay = 158;
                                else if (!IsRear && IsRearUnit) delay = 218;
                                else if (IsRear && IsRearUnit) delay = 346;
                                if (CurrentPowerMode == PowerMode.Diesel && !IsRear) delay += 50;
                                LocomotivePositionDelay.Setup(delay);
                                LocomotivePositionDelay.Start();
                                CurrentTractionSystemChangeState = TractionSystemChangeState.WaitNeutral;
                                if (CurrentPowerMode != PowerMode.Diesel) TargetPowerMode = CurrentPowerMode == PowerMode.DC3kV ? PowerMode.AC25kV : PowerMode.DC3kV;
                            }
                            break;
                        case 4:
                            if (!PantoUp && TargetPowerMode == PowerMode.DC3kV) TargetPowerMode = PowerMode.AC25kV;
                            else if (!PantoUp && TargetPowerMode == PowerMode.AC25kV) SimulatedVoltage = 25000;
                            break;
                        case 5:
                            if (!PantoUp && TargetPowerMode == PowerMode.AC25kV) TargetPowerMode = PowerMode.DC3kV;
                            else if (!PantoUp && TargetPowerMode == PowerMode.DC3kV) SimulatedVoltage = 3000;
                            break;
                        case 8:
                            CurrentTractionSystemChangeState = TractionSystemChangeState.AuxiliaryFunction;
                            break;
                    }
                    break;
                default:
                    base.HandleEventFromLeadLocomotive(evt, id);
                    break;
            } 
        }

        public override void HandleEventFromOtherLocomotive(int locoIndex, PowerSupplyEvent evt, int id)
        {
            switch(evt)
            {
                case PowerSupplyEvent.CloseCircuitBreaker:
                    CircuitBreakerState cbstate = (CircuitBreakerState)(id&3);
                    id /= 4;
                    if (locoIndex == RemoteHeadIndex)
                    {
                        CircuitBreakerState tlstate = (CircuitBreakerState)(id&3);
                        id /= 4;
                        CircuitBreakerState rlstate = (CircuitBreakerState)(id&3);
                        id /= 4;
                        RemoteCircuitBreakerState = cbstate;
                        RemoteTrainLineContactorState = tlstate;
                        RemoteRoofLineContactorState = rlstate;
                    }
                    else if (DualMode)
                    {
                        PowerSupplyState generator = (id&1) == 1 ? PowerSupplyState.PowerOn : PowerSupplyState.PowerOff;
                        id /= 2;
                        if (locoIndex == LocalCETIndex)
                        {
                            LocalCETCircuitBreakerState = cbstate;
                            LocalCETGeneratorState = generator;
                        }
                        else if (locoIndex == RemoteCETIndex)
                        {
                            RemoteCETCircuitBreakerState = cbstate;
                            RemoteCETGeneratorState = generator;
                        }
                    }
                    break;
                case PowerSupplyEvent.RaisePantograph:
                    if (locoIndex == RemoteHeadIndex) RemotePantographVoltageV = id;
                    else if (locoIndex == LocalCETIndex) LocalCETPowerW = id;
                    else if (locoIndex == RemoteCETIndex) RemoteCETPowerW = id;
                    break;
                // > 0: series identifier, stop if incompatible loco
                // <= 0: -index of compatible loco
                case PowerSupplyEvent.SwitchOnElectricTrainSupply:
                    if (id == 130 && locoIndex != IndexOfLocomotive())
                    {
                        prevNumberOfLocomotives = NumberOfLocomotives();
                        ConnectedLocomotives.Clear();
                        ConnectedLocomotives[IndexOfLocomotive()] = 130;
                        SignalEventToOtherLocomotive(locoIndex, PowerSupplyEvent.SwitchOffElectricTrainSupply, -1);
                        int tryNextIndex = -1;
                        if (locoIndex < IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() + 1;
                        else if (locoIndex > IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() - 1;
                        if (tryNextIndex >= 0 && tryNextIndex < NumberOfLocomotives())
                        {
                            SignalEventToOtherLocomotive(tryNextIndex, evt, id);
                        }
                        SignalEventToOtherLocomotive(locoIndex, evt, -IndexOfLocomotive());
                    }
                    else if (id <= 0)
                    {
                        int targetIndex = -id;
                        if (!ConnectedLocomotives.ContainsKey(targetIndex))
                        {
                            SignalEventToOtherLocomotive(targetIndex, PowerSupplyEvent.SwitchOffElectricTrainSupply, -1);
                        }
                        int tryNextIndex = -1;
                        if (locoIndex < IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() + 1;
                        else if (locoIndex > IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() - 1;
                        if (ConnectedLocomotives.ContainsKey(tryNextIndex))
                        {
                            SignalEventToOtherLocomotive(tryNextIndex, evt, id);
                        }
                    }
                    break;
                case PowerSupplyEvent.SwitchOffElectricTrainSupply:
                    if (id == -1)
                    {
                        SignalEventToOtherLocomotive(locoIndex, evt, 130);
                        if (!ConnectedLocomotives.ContainsKey(locoIndex)) SignalEventToOtherLocomotive(locoIndex, evt, -1);
                    }
                    else ConnectedLocomotives[locoIndex] = id;
                    break;
            }
        }

        public override void HandleEventFromTcs(PowerSupplyEvent evt)
        {
            switch(evt)
            {
                case PowerSupplyEvent.OpenCircuitBreaker:

                    if (CurrentPowerMode != PowerMode.Diesel && CurrentNeutralSectionState == NeutralSectionState.Inactive)
                    {
                        LocomotivePositionDelay.Setup(SpeedMpS() * 6);
                        LocomotivePositionDelay.Start();
                        CurrentNeutralSectionState = NeutralSectionState.Preparation;
                        NeutralSectionByTCS = true;
                        SignalCustomEvent(CustomEvent.StartNeutralSectionTCS);
                    }
                    break;
                case PowerSupplyEvent.CloseCircuitBreaker:
                    {
                        LocomotivePositionDelay.Setup(30);
                        LocomotivePositionDelay.Start();
                        SignalCustomEvent(CustomEvent.EndNeutralSectionTCS);
                    }
                    break;
                default:
                    base.HandleEventFromTcs(evt);
                    break;
            }
        }
        public override void HandleEventFromTcs(PowerSupplyEvent evt, string msg)
        {
            switch(evt)
            {
                case PowerSupplyEvent.LowerPantograph:
                    LocomotivePositionDelay.Setup(0);
                    LocomotivePositionDelay.Start();
                    CurrentPantographDownSectionState = PantographDownSectionState.Preparation;
                    SignalCustomEvent(CustomEvent.LowerPantographTCS);
                    break;
                case PowerSupplyEvent.RaisePantograph:
                    if (CurrentPantographDownSectionState == PantographDownSectionState.Active)
                    {
                        LocomotivePositionDelay.Setup(30);
                        LocomotivePositionDelay.Start();
                        SignalCustomEvent(CustomEvent.RaisePantographTCS);
                    }
                    break;
                default:
                    base.HandleEventFromTcs(evt, msg);
                    break;
            }
        }
        
        float SimulatedVoltage;
        OdoMeter RestoreVoltageOdometer;
        public float SimulatedLineVoltageV()
        {
            if (SimulatedVoltage == 0) SimulatedVoltage = LineVoltageV();
            if (CurrentTractionSystemChangeState >= TractionSystemChangeState.Active) SimulatedVoltage = TargetPowerMode == PowerMode.DC3kV ? 3000 : 25000;
            if (CurrentPantographDownSectionState == PantographDownSectionState.Active) SimulatedVoltage = TargetPowerMode == PowerMode.DC3kV ? 3000 : 25000;
            if (CurrentNeutralSectionState == NeutralSectionState.Active || CurrentTractionSystemChangeState == TractionSystemChangeState.Active) return 0;
            if (CurrentNeutralSectionState == NeutralSectionState.WaitVoltage || CurrentTractionSystemChangeState == TractionSystemChangeState.WaitVoltage)
            {
                if (RestoreVoltageOdometer.Triggered)
                {
                    return SimulatedVoltage;
                }
                if (!RestoreVoltageOdometer.Started)
                {
                    if (CurrentNeutralSectionState == NeutralSectionState.WaitVoltage) RestoreVoltageOdometer.Setup(400.0f);
                    else RestoreVoltageOdometer.Setup(50.0f);
                    RestoreVoltageOdometer.Start();
                }
                return 0;
            }
            else if (RestoreVoltageOdometer.Started) RestoreVoltageOdometer.Stop();
            return SimulatedVoltage;
        }
    }
}
