﻿using System;
using Orts.Common;
using ORTS.Scripting.Api;
using Orts.Simulation;

namespace ORTS.Scripting.Script
{
    public class T250CircuitBreaker : CircuitBreaker
    {
        private Timer ClosingTimer;
        private CircuitBreakerState PreviousState;
        private bool QuickPowerOn;
        private bool SupplyClosingAuthorization;

        private CircuitBreakerState LocalCircuitBreakerState;
        private CircuitBreakerState RemoteCircuitBreakerState;

        enum PowerMode
        {
            AC25kV,
            DC3kV,
            Diesel,
        }
        private PowerMode CurrentPowerMode;
        private bool LocalPantoUp;

        public override void Initialize()
        {
            ClosingTimer = new Timer(this);
            ClosingTimer.Setup(ClosingDelayS());

            SetDriverClosingAuthorization(true);
        }

        public override void Update(float elapsedSeconds)
        {
            bool closingAuthorization = TCSClosingAuthorization() && LocalPantoUp;
            if (CurrentPowerMode == PowerMode.AC25kV)
            {
                closingAuthorization &= PantographVoltageV > 16000 && PantographVoltageV < 31000;
            }
            else if (CurrentPowerMode == PowerMode.DC3kV)
            {
                closingAuthorization &= PantographVoltageV > 1900 && PantographVoltageV < 4250;
            }
            else
            {
                closingAuthorization = false;
            }
            SetClosingAuthorization(closingAuthorization);

            switch (LocalCircuitBreakerState)
            {
                case CircuitBreakerState.Closed:
                    if (!ClosingAuthorization() || DriverOpeningOrder())
                    {
                        LocalCircuitBreakerState = CircuitBreakerState.Open;
                    }
                    break;

                case CircuitBreakerState.Closing:
                    if (ClosingAuthorization() && (DriverClosingOrder() || QuickPowerOn))
                    {
                        if (!ClosingTimer.Started)
                        {
                            ClosingTimer.Start();
                        }

                        if (ClosingTimer.Triggered)
                        {
                            QuickPowerOn = false;
                            ClosingTimer.Stop();
                            LocalCircuitBreakerState = CircuitBreakerState.Closed;
                        }
                    }
                    else
                    {
                        QuickPowerOn = false;
                        ClosingTimer.Stop();
                        LocalCircuitBreakerState = CircuitBreakerState.Open;
                    }
                    break;

                case CircuitBreakerState.Open:
                    if (ClosingAuthorization() && (DriverClosingOrder() || QuickPowerOn))
                    {
                        LocalCircuitBreakerState = CircuitBreakerState.Closing;
                    }
                    else if (QuickPowerOn)
                    {
                        QuickPowerOn = false;
                    }
                    break;
            }

            if (PreviousState != LocalCircuitBreakerState)
            {
                switch (LocalCircuitBreakerState)
                {
                    case CircuitBreakerState.Open:
                        SignalEvent(Event.CircuitBreakerOpen);
                        SignalEventToPowerSupply(PowerSupplyEvent.CloseCircuitBreaker, 0);
                        break;

                    case CircuitBreakerState.Closing:
                        SignalEvent(Event.CircuitBreakerClosing);
                        SignalEventToPowerSupply(PowerSupplyEvent.CloseCircuitBreaker, 1);
                        break;

                    case CircuitBreakerState.Closed:
                        SignalEvent(Event.CircuitBreakerClosed);
                        SignalEventToPowerSupply(PowerSupplyEvent.CloseCircuitBreaker, 2);
                        break;
                }
            }

            PreviousState = LocalCircuitBreakerState;

            if (RemoteCircuitBreakerState > LocalCircuitBreakerState) SetCurrentState(RemoteCircuitBreakerState);
            else SetCurrentState(LocalCircuitBreakerState);
        }

        public override void HandleEvent(PowerSupplyEvent evt)
        {
            switch (evt)
            {
                case PowerSupplyEvent.CloseCircuitBreakerButtonPressed:
                    SetDriverOpeningOrder(false);
                    SetDriverClosingOrder(true);
                    SignalEvent(Event.CircuitBreakerClosingOrderOn);
                    if (ClosingAuthorization())
                        Confirm(CabControl.CircuitBreakerClosingOrder, CabSetting.On);
                    break;
                case PowerSupplyEvent.CloseCircuitBreakerButtonReleased:
                    SetDriverClosingOrder(false);
                    break;
                case PowerSupplyEvent.OpenCircuitBreakerButtonPressed:
                    SetDriverClosingOrder(false);
                    SetDriverOpeningOrder(true);
                    QuickPowerOn = false;
                    SignalEvent(Event.CircuitBreakerClosingOrderOff);
                    Confirm(CabControl.CircuitBreakerClosingOrder, CabSetting.Off);
                    break;
                case PowerSupplyEvent.OpenCircuitBreakerButtonReleased:
                    SetDriverOpeningOrder(false);
                    break;
                case PowerSupplyEvent.QuickPowerOn:
                    QuickPowerOn = true;
                    break;
                case PowerSupplyEvent.QuickPowerOff:
                    QuickPowerOn = false;
                    LocalCircuitBreakerState = CircuitBreakerState.Open;
                    break;
            }
        }

        public override void HandleEvent(PowerSupplyEvent evt, int id)
        {
            switch (evt)
            {
                case PowerSupplyEvent.CloseCircuitBreaker:
                    RemoteCircuitBreakerState = (CircuitBreakerState)(id&3);
                    id /= 4;
                    CurrentPowerMode = (PowerMode)(id&3);
                    id /= 4;
                    LocalPantoUp = (id&1)==1;
                    id /= 2;
                    break;
            }
        }
    }
}
