using System;
using ORTS.Common;
using ORTS.Scripting.Api;
using Orts.Simulation;
using Event = Orts.Common.Event;
using System.Collections.Generic;
namespace ORTS.Scripting.Script
{
    public class T250CETPowerSupply : DieselPowerSupply
    {
        private Timer AuxPowerOnTimer;
        private Timer DieselEngineStopTimer;

        private DieselEngineState PreviousEngineState;

        bool QuickPowerOn = false;

        bool DieselMode = false;

        int LeadIndex;

        bool IsRear;
        bool Flipped;
        bool IsRearUnit;
        bool IsMultipleUnit;
        int LocalHeadIndex;
        int RemoteHeadIndex;

        enum PowerMode
        {
            AC25kV,
            DC3kV,
            Diesel,
        }
        PowerMode TargetPowerMode;
        PowerMode CurrentPowerMode;

        SortedDictionary<int,int> ConnectedLocomotives = new SortedDictionary<int,int>();

        public override void Initialize()
        {
            AuxPowerOnTimer = new Timer(this);
            AuxPowerOnTimer.Setup(AuxPowerOnDelayS());

            DieselEngineStopTimer = new Timer(this);
            DieselEngineStopTimer.Setup(5*60);

            PreviousEngineState = CurrentDieselEngineState(0);
        }

        public override void Update(float elapsedClockSeconds)
        {
            {
                int headno=0;
                int cetno=0;
                int unitno=0;
                int ownunit=-1;
                int ownno=-1;
                int leadunit=-1;
                int leadno=-1;
                foreach (var kvp in ConnectedLocomotives)
                {
                    if (headno > 1)
                    {
                        unitno++;
                        headno = cetno = 0;
                    }
                    if (kvp.Key == LeadIndex)
                    {
                        leadunit = unitno;
                        leadno = headno;
                    }
                    if (kvp.Key == IndexOfLocomotive())
                    {
                        ownunit = unitno;
                        ownno = cetno;
                        Flipped = cetno == 1;
                    }
                    if (kvp.Value == 130) headno++;
                    else if (kvp.Value == 730) cetno++;
                }
                IsRear = leadno != ownno;
                IsRearUnit = ownunit != leadunit;
                IsMultipleUnit = unitno > 0;
                headno=0;
                cetno=0;
                unitno=0;
                LocalHeadIndex = -1;
                RemoteHeadIndex = -1;
                foreach (var kvp in ConnectedLocomotives)
                {
                    if (headno > 1)
                    {
                        unitno++;
                        headno = cetno = 0;
                    }
                    if (unitno == ownunit)
                    {
                        if (kvp.Value == 130)
                        {
                            if (ownno == headno) LocalHeadIndex = kvp.Key;
                            else RemoteHeadIndex = kvp.Key;
                        }
                    }
                    if (kvp.Value == 130) headno++;
                    else if (kvp.Value == 730) cetno++;
                }
            }
            SetCurrentBatteryState(BatterySwitchOn() ? PowerSupplyState.PowerOn : PowerSupplyState.PowerOff);
            SetCurrentLowVoltagePowerSupplyState(BatterySwitchOn() ? PowerSupplyState.PowerOn : PowerSupplyState.PowerOff);
            SetCurrentCabPowerSupplyState(PowerSupplyState.PowerOff);
            SetCurrentMainPowerSupplyState(PowerSupplyState.PowerOff);

            float minRPM = 0;

            if (TargetPowerMode != PowerMode.Diesel|| CurrentDieselEnginesState() != DieselEngineState.Running)
            {
                if (CurrentAuxiliaryPowerSupplyState() != PowerSupplyState.PowerOff)
                {
                    if (CurrentTractionCutOffRelayState() != TractionCutOffRelayState.Open)
                    {
                        SignalEventToTractionCutOffRelay(PowerSupplyEvent.QuickPowerOff);
                    }
                    else
                    {
                        SetCurrentAuxiliaryPowerSupplyState(PowerSupplyState.PowerOff);
                        SignalEvent(Event.PowerConverterOff);
                    }
                }
                if (AuxPowerOnTimer.Started)
                    AuxPowerOnTimer.Stop();
                if (QuickPowerOn && (CurrentDieselEnginesState() == DieselEngineState.Stopped || (CurrentDieselEnginesState() == DieselEngineState.Running && TargetPowerMode != PowerMode.Diesel)))
                    QuickPowerOn = false;
            }
            else
            {
                if (!AuxPowerOnTimer.Started)
                    AuxPowerOnTimer.Start();
                minRPM = 800;
                if (AuxPowerOnTimer.Triggered && CurrentAuxiliaryPowerSupplyState() == PowerSupplyState.PowerOff)
                {
                    SignalEvent(Event.PowerConverterOn);
                    SetCurrentAuxiliaryPowerSupplyState(PowerSupplyState.PowerOn);
                    if (QuickPowerOn && CurrentTractionCutOffRelayState() == TractionCutOffRelayState.Open)
                    {
                        SignalEventToTractionCutOffRelay(PowerSupplyEvent.QuickPowerOn);
                        QuickPowerOn = false;
                    }
                }
            }

            if (CurrentAuxiliaryPowerSupplyState() != PowerSupplyState.PowerOn && CurrentTractionCutOffRelayState() != TractionCutOffRelayState.Open)
            {
                SignalEventToTractionCutOffRelay(PowerSupplyEvent.QuickPowerOff);
            }

            SetCurrentDynamicBrakeAvailability(false);

            int state = ((int)CurrentTractionCutOffRelayState()) + (CurrentAuxiliaryPowerSupplyState() == PowerSupplyState.PowerOn ? 1 : 0)*4;
            SignalEventToOtherLocomotive(RemoteHeadIndex, PowerSupplyEvent.CloseCircuitBreaker, state);
            SignalEventToOtherLocomotive(LocalHeadIndex, PowerSupplyEvent.CloseCircuitBreaker, state);

            float availableTractionPowerW = Math.Max(DieselEngineOutputPowerW*0.95f - ElectricTrainSupplyPowerW, 0);
            SignalEventToOtherLocomotive(LocalHeadIndex, PowerSupplyEvent.RaisePantograph, (int)availableTractionPowerW);

            if (CurrentTractionCutOffRelayState() == TractionCutOffRelayState.Closed && ElectricTrainSupplySwitchOn())
            {
                SetCurrentElectricTrainSupplyState(PowerSupplyState.PowerOn);
                float targetForcekN;
                if (SpeedMpS() < MpS.FromKpH(50)) targetForcekN = 110-SpeedMpS()/MpS.FromKpH(50)*30;
                else if (SpeedMpS() < 1400.0f/80) targetForcekN = 80;
                else targetForcekN = 1400.0f/SpeedMpS();
                targetForcekN *= ThrottlePercent()/100.0f;
                float targetPowerW = Math.Max(SpeedMpS(), MpS.FromKpH(10))*targetForcekN*1000*1.1f;
                targetPowerW += 3.0e5f+ElectricTrainSupplyPowerW;
                targetPowerW = Math.Max(targetPowerW-2.0e5f, 0);
                minRPM = Math.Min(800 + targetPowerW/1.6e6f*1000, 1800);
            }
            else
            {
                SetCurrentElectricTrainSupplyState(PowerSupplyState.PowerOff);
            }
            DieselEngineMinRpm = minRPM;

            if (DieselEngineStopTimer.Triggered)
            {
                DieselEngineStopTimer.Stop();
                SignalEventToDieselEngines(PowerSupplyEvent.StopEngine);
            }

            UpdateSounds();
        }

        protected void UpdateSounds()
        {
            // First engine
            if ((PreviousEngineState == DieselEngineState.Stopped
                || PreviousEngineState == DieselEngineState.Stopping)
                && (CurrentDieselEngineState(0) == DieselEngineState.Starting
                || CurrentDieselEngineState(0) == DieselEngineState.Running))
            {
                SignalEvent(Event.EnginePowerOn);
            }
            else if ((PreviousEngineState == DieselEngineState.Starting
                || PreviousEngineState == DieselEngineState.Running)
                && (CurrentDieselEngineState(0) == DieselEngineState.Stopping
                || CurrentDieselEngineState(0) == DieselEngineState.Stopped))
            {
                SignalEvent(Event.EnginePowerOff);
            }
            PreviousEngineState = CurrentDieselEngineState(0);
        }

        public override void HandleEvent(PowerSupplyEvent evt)
        {
            switch (evt)
            {
                case PowerSupplyEvent.QuickPowerOn:
                case PowerSupplyEvent.QuickPowerOff:
                    bool poweroff = (CurrentDieselEnginesState() != DieselEngineState.Stopped && CurrentDieselEnginesState() != DieselEngineState.Stopping) || evt == PowerSupplyEvent.QuickPowerOff;
                    if (poweroff)
                    {
                        QuickPowerOn = false;
                        SignalEventToElectricTrainSupplySwitch(PowerSupplyEvent.SwitchOffElectricTrainSupply);
                        SignalEventToTractionCutOffRelay(PowerSupplyEvent.QuickPowerOff);
                        if (!DieselEngineStopTimer.Started) DieselEngineStopTimer.Start();
                        SignalEventToBatterySwitch(PowerSupplyEvent.QuickPowerOff);
                    }
                    else
                    {
                        QuickPowerOn = true;
                        SignalEventToBatterySwitch(PowerSupplyEvent.QuickPowerOn);
                        SignalEventToDieselEngines(PowerSupplyEvent.StartEngine);
                        DieselEngineStopTimer.Stop();
                        SignalEventToElectricTrainSupplySwitch(PowerSupplyEvent.SwitchOnElectricTrainSupply);
                    }
                    break;

                default:
                    base.HandleEvent(evt);
                    break;
            }
        }
        public override void HandleEventFromLeadLocomotive(PowerSupplyEvent evt)
        {
            switch (evt)
            {
                case PowerSupplyEvent.StartEngine:
                    SignalEventToDieselEngines(evt);
                    DieselEngineStopTimer.Stop();
                    break;
                case PowerSupplyEvent.StopEngine:
                    if (!DieselEngineStopTimer.Started) DieselEngineStopTimer.Start();
                    break;
                default:
                    base.HandleEvent(evt);
                    break;
            }
        }
        public override void HandleEventFromLeadLocomotive(PowerSupplyEvent evt, int id)
        {
            switch (evt)
            {
                case PowerSupplyEvent.TurnOnMasterKey:
                    LeadIndex = id;
                    break;
                case PowerSupplyEvent.IncreaseVoltageSelectorPosition:
                    {
                        CurrentPowerMode = (PowerMode)(id&3);
                        id /= 4;
                        TargetPowerMode = (PowerMode)(id&3);
                    }
                    break;
                default:
                    base.HandleEvent(evt, id);
                    break;
            }
        }
        public override void HandleEventFromOtherLocomotive(int locoIndex, PowerSupplyEvent evt, int id)
        {
            switch(evt)
            {
                // > 0: series identifier, stop if incompatible loco
                // <= 0: -index of compatible loco
                case PowerSupplyEvent.SwitchOnElectricTrainSupply:
                    if (id == 130 && locoIndex != IndexOfLocomotive())
                    {
                        ConnectedLocomotives.Clear();
                        ConnectedLocomotives[IndexOfLocomotive()] = 730;
                        SignalEventToOtherLocomotive(locoIndex, PowerSupplyEvent.SwitchOffElectricTrainSupply, -1);
                        int tryNextIndex = -1;
                        if (locoIndex < IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() + 1;
                        else if (locoIndex > IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() - 1;
                        if (tryNextIndex >= 0 && tryNextIndex < NumberOfLocomotives())
                        {
                            SignalEventToOtherLocomotive(tryNextIndex, evt, id);
                        }
                        SignalEventToOtherLocomotive(locoIndex, evt, -IndexOfLocomotive());
                    }
                    else if (id <= 0)
                    {
                        int targetIndex = -id;
                        if (!ConnectedLocomotives.ContainsKey(targetIndex))
                        {
                            SignalEventToOtherLocomotive(targetIndex, PowerSupplyEvent.SwitchOffElectricTrainSupply, -1);
                        }
                        int tryNextIndex = -1;
                        if (locoIndex < IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() + 1;
                        else if (locoIndex > IndexOfLocomotive()) tryNextIndex = IndexOfLocomotive() - 1;
                        if (ConnectedLocomotives.ContainsKey(tryNextIndex))
                        {
                            SignalEventToOtherLocomotive(tryNextIndex, evt, id);
                        }
                    }
                    break;
                case PowerSupplyEvent.SwitchOffElectricTrainSupply:
                    if (id == -1)
                    {
                        SignalEventToOtherLocomotive(locoIndex, evt, 730);
                        if (!ConnectedLocomotives.ContainsKey(locoIndex)) SignalEventToOtherLocomotive(locoIndex, evt, -1);
                    }
                    else ConnectedLocomotives[locoIndex] = id;
                    break;
            }
        }
    }
}
